﻿using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Linq;

using HIPS.CommonSchemas;
using HIPS.Web.ModelInterface.Common;
using HIPS.Web.Components.ServiceModel;

using HIPS.Web.Test.Common;

namespace HIPS.Web.Test.PcehrView.Repository
{

    /// <summary>
    /// Unit tests for the PatientRepository.
    /// </summary>
    /// <history>
    ///   <change user="David Sampson (Chamonix)" date="25 November 2013">Initial version.</change>
    /// </history>
    [TestClass]
    public class PatientRepositoryTests : TestBase
    {

        #region Methods

        #region Tests

        /// <summary>
        /// Test that we can retrieve a list of patients with a PCEHR.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="25 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Patients_With_Pcehr()
        {
            IPatientRepository repo = null;
            try
            {
                //Arrange
                string hospitalCode = "RCH";
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository();

                //Act
                var patients = repo.ListPatientsCurrentlyInHospital(
                    hospitalCodeSystem: TestAssistant.HOSPITAL_CODE_SYSTEM,
                    hospitalCode: hospitalCode, 
                    withPcehr: true,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(patients, "Test for non-null result failed.");
                Assert.AreEqual(HipsResponseIndicator.OK, patients.Data.HipsResponse.Status, "Test for OK response status failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.Count > 0, "Test for non-zero count of patients failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.All(p => p.HospitalCode == hospitalCode), "Test for matching hospital code failed.");
                Assert.IsTrue(
                    patients.Data.PatientInHospitalList.All(p => (p.ParticipationStatus == ConsentSchemas.ParticipationStatus.PcehrAdvertised) || (p.ParticipationStatus == ConsentSchemas.ParticipationStatus.RequestedUpload)),
                    "Test for matching PCEHR status failed");
                Assert.IsTrue(patients.Data.PatientInHospitalList.All(p => ! p.DischargeDate.HasValue), "Test for non-discharge failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can retrieve a list of patients without a PCEHR.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="25 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Patients_Without_Pcehr()
        {
            IPatientRepository repo = null;
            try
            {
                //Arrange
                string hospitalCode = "RCH";
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository();

                //Act
                var patients = repo.ListPatientsCurrentlyInHospital(
                    hospitalCodeSystem: TestAssistant.HOSPITAL_CODE_SYSTEM,
                    hospitalCode: hospitalCode,
                    withPcehr: false,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(patients, "Test for non-null result failed.");
                Assert.AreEqual(HipsResponseIndicator.OK, patients.Data.HipsResponse.Status, "Test for OK response status failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.Count > 0, "Test for non-zero count of patients failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.All(p => p.HospitalCode == hospitalCode), "Test for matching hospital code failed.");
                Assert.IsTrue(
                    patients.Data.PatientInHospitalList.All(p => p.ParticipationStatus == ConsentSchemas.ParticipationStatus.PcehrNotAdvertised),
                    "Test for matching PCEHR status failed");
                Assert.IsTrue(patients.Data.PatientInHospitalList.All(p => !p.DischargeDate.HasValue), "Test for non-discharge failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can retrieve a list of patients (regardless of whether they have a PCEHR).
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="25 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Patients_All()
        {
            IPatientRepository repo = null;
            try
            {
                //Arrange
                string hospitalCode = "RCH";
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository();

                //Act
                var patients = repo.ListPatientsCurrentlyInHospital(
                    hospitalCodeSystem: TestAssistant.HOSPITAL_CODE_SYSTEM,
                    hospitalCode: hospitalCode,
                    withPcehr: null,
                    operatingUser: operatingUser);

                //Assert
                Assert.IsNotNull(patients, "Test for non-null result failed.");
                Assert.AreEqual(HipsResponseIndicator.OK, patients.Data.HipsResponse.Status, "Test for OK response status failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.Count > 0, "Test for non-zero count of patients failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.All(p => p.HospitalCode == hospitalCode), "Test for matching hospital code failed.");
                Assert.IsTrue(patients.Data.PatientInHospitalList.All(p => !p.DischargeDate.HasValue), "Test for non-discharge failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        /// <summary>
        /// Test that we can retrieve a list of patients (regardless of whether they have a PCEHR), using caching.
        /// </summary>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="28 November 2013">Initial version.</change>
        /// </history>
        [TestMethod]
        public void Can_List_Patients_All_From_Cache()
        {
            IPatientRepository repo = null;
            try
            {
                //Arrange
                string hospitalCode = "RCH";
                UserDetails operatingUser = TestAssistant.GetCurrentUserDetails();

                repo = this.GetRepository(true);

                System.Func<ServiceResponse<PatientSchemas.PatientListResponse>> operation = () =>
                    {
                        var sw = System.Diagnostics.Stopwatch.StartNew();
                        var result = repo.ListPatientsCurrentlyInHospital(
                            hospitalCodeSystem: TestAssistant.HOSPITAL_CODE_SYSTEM,
                            hospitalCode: hospitalCode,
                            withPcehr: null,
                            operatingUser: operatingUser);
                        sw.Stop();
                        this.TestContext.WriteLine("Elapsed time: {0}.", sw.Elapsed.ToString());
                        return result;
                    };
                
                //Act
                var result1 = operation.Invoke();
                var result2 = operation.Invoke();

                //Assert
                Assert.IsNotNull(result1, "Test for non-null result failed.");
                Assert.IsNotNull(result2, "Test for non-null result failed.");
            }
            finally
            {
                if (repo != null)
                {
                    repo.Dispose();
                }
            }
        }

        #endregion

        #region Helpers

        /// <summary>
        /// Gets the repository to be tested.
        /// </summary>
        /// <param name="withCaching">Boolean value indicating whether to employ caching in the repository.</param>
        /// <returns>Repository instance.</returns>
        /// <history>
        ///   <change user="David Sampson (Chamonix)" date="25 November 2013">Initial version.</change>
        /// </history>
        private IPatientRepository GetRepository(bool withCaching = false)
        {
            if (withCaching)
            {
                return new HIPS.Web.Data.Hips.PatientRepository(0, new HIPS.Web.Components.Cache.MemoryCacheProvider(TimeSpan.FromMinutes(5)));
            }
            else
            {
                return new HIPS.Web.Data.Hips.PatientRepository(0, new HIPS.Web.Components.Cache.NoCachingCacheProvider());
            }
        }

        #endregion

        #endregion

    }
}
